<?php
namespace App\Http\Controllers\MobileApp;

use App\Models\Brand;
use App\Models\Vendor;
use App\Models\Category;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Models\CustomPartRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Http\Resources\BrandResource;
use Illuminate\Support\Facades\Cache;
use App\Http\Resources\VendorResource;
use App\Http\Resources\CategoryResource;
use Illuminate\Validation\ValidationException;
use App\Http\Resources\CustomPartRequestResource;
use App\Http\Requests\MobileApi\StoreCustomPartRequest;
use App\Notifications\NewCustomPartRequestNotification;

class CustomPartRequestController extends Controller
{
    /**
     * Constructor to set locale
     */
    public function __construct(Request $request)
    {
        $locale = $request->header('Accept-Language', 'ar');
        \Illuminate\Support\Facades\App::setLocale($locale);
    }

    public function store(Request $request)
    {
        try {
            // Validate the request
            $validated = $request->validate([
                'model_id' => 'required|exists:models,id',
                'category_id' => 'required|exists:categories,id',
                'vendor_id' => 'required|exists:vendors,id',
                'description' => 'required|string',
                'address' => 'required|string',
                'items' => 'required|array|min:1',
                'items.*.name' => 'required|string',
                'items.*.quantity' => 'required|integer|min:1',
                'items.*.notes' => 'nullable|string',
                'items.*.image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            ]);

            // Check for guest user
            $userId = Auth::guard('sanctum')->check() ? Auth::guard('sanctum')->user()->id : null;
            $tokenId = null;
            if (!$userId) {
                $tokenId = $request->header('tokenID');
                if (!$tokenId) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'يجب توفير tokenID في الهيدر للمستخدمين الغير مصرح لهم',
                    ], 400);
                }
            }

            // Prepare request data
            $requestData = [
                'model_id' => $validated['model_id'],
                'category_id' => $validated['category_id'],
                'vendor_id' => $validated['vendor_id'],
                'description' => $validated['description'],
                'status' => 'pending',
                'address' => $validated['address'],
                'user_id' => $userId,
                'code' => $tokenId,
            ];

            DB::beginTransaction();

            // Create the custom part request
            $customPartRequest = CustomPartRequest::create($requestData);

            // Process each item and handle image uploads
            foreach ($validated['items'] as $index => $itemData) {
                $imagePath = null;
                if ($request->hasFile("items.{$index}.image")) {
                    $imagePath = $request->file("items.{$index}.image")->store('images/parts', 'public');
                }

                $customPartRequest->items()->create([
                    'name' => $itemData['name'],
                    'quantity' => $itemData['quantity'],
                    'notes' => $itemData['notes'],
                    'image' => $imagePath,
                ]);
            }

            // Send notification to vendor
            $vendor = $customPartRequest->vendor;
            // if ($vendor) {
            //     $vendor->notify(new NewCustomPartRequestNotification([
            //         'custom_part_request_id' => $customPartRequest->id,
            //         'message' => trans('messages.custom_part_created'),
            //     ]));
            // }

            DB::commit();

            // Load relationships for response
            $customPartRequest->load(['model', 'category', 'vendor', 'items']);

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.custom_part_request_created'),
                'data' => new CustomPartRequestResource($customPartRequest),
                'settings' => [
                    'welcome_message' => 'hello',
                    'color_top_header' => 'part mart',
                ],
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating custom part request: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_create_custom_part_request'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function index(Request $request)
    {
        try {
            $userId = auth()->id();
            if (!$userId) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            $requests = CustomPartRequest::with(['model', 'category', 'vendor', 'items'])
                ->where('user_id', $userId)
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.custom_part_requests_retrieved'),
                'data' => CustomPartRequestResource::collection($requests),
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error retrieving custom part requests: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_custom_part_requests'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function data()
    {
        $vendors = Vendor::where('type', 2)->get();
        return response()->json([
            'status' => 'success',
            'service_center' => VendorResource::collection($vendors),
            'brands' => BrandResource::collection(Brand::all()),
            'category' => CategoryResource::collection(Category::all())

        ], 200);
    }
}