<?php
namespace App\Http\Controllers\MobileApp;

use App\Http\Controllers\Controller;
use App\Http\Requests\MobileApi\StoreCarForSaleRequest;
use App\Http\Resources\CarForSaleResource;
use App\Models\CarForSale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CarForSaleController extends Controller
{
    public function index(Request $request)
    {
        try {
            $locale = app()->getLocale();
            $cacheKey = 'cars_for_sale_' . md5(serialize($request->all()) . '_' . $locale);

            $cars = Cache::remember($cacheKey, 3600, function () use ($request) {
                $query = CarForSale::with(['model', 'vendor'])
                    ->whereNull('deleted_at');

                // Apply filters
                if ($request->has('model_id')) {
                    $query->where('model_id', $request->input('model_id'));
                }
                if ($request->has('min_price')) {
                    $query->where('price', '>=', $request->input('min_price'));
                }
                if ($request->has('max_price')) {
                    $query->where('price', '<=', $request->input('max_price'));
                }
                if ($request->has('type')) {
                    $query->where('type', $request->input('type'));
                }
                if ($request->has('transmission_type')) {
                    $query->where('transmission_type', $request->input('transmission_type'));
                }
                if ($request->has('production_year')) {
                    $query->where('production_year', $request->input('production_year'));
                }
                if ($request->has('installment_available')) {
                    $query->where('installment_available', $request->input('installment_available'));
                }

                return $query->orderBy('created_at', 'desc')
                    ->paginate(10);
            });

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.cars_for_sale_retrieved'),
                'data' => CarForSaleResource::collection($cars),
                'pagination' => [
                    'current_page' => $cars->currentPage(),
                    'last_page' => $cars->lastPage(),
                    'per_page' => $cars->perPage(),
                    'total' => $cars->total(),
                ],
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error retrieving cars for sale', [
                'query' => $request->all(),
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_cars_for_sale'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    public function store(StoreCarForSaleRequest $request)
    {
        try {
            $user = auth()->guard('sanctum')->user();

            $data = $request->validated();
            $data['user_id'] = $user->id;

            DB::beginTransaction();
            $car = CarForSale::create($data);
            DB::commit();


            return response()->json([
                'status' => 'success',
                'message' => trans('messages.car_for_sale_created'),
                'data' => new CarForSaleResource($car->load(['model'])),
                // 'settings' => [
                //     'welcome_message' => $settings->welcome_message,
                //     'color_top_header' => $settings->color_top_header,
                // ],
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating car for sale', [
                'user_id' => $user->id ?? null,
                'data' => $request->all(),
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_create_car_for_sale'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $car = CarForSale::with(['model', 'vendor'])
                ->whereNull('deleted_at')
                ->findOrFail($id);

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.car_for_sale_retrieved'),
                'data' => new CarForSaleResource($car),
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.car_for_sale_not_found'),
            ], 404);
        } catch (\Exception $e) {
            Log::error('Error retrieving car for sale', [
                'car_id' => $id,
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_car_for_sale'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }
}